<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\web\UploadedFile;
use yii\web\NotFoundHttpException;
use app\models\EmployeeProfile;
use app\models\User;

class ProfileController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            return !Yii::$app->user->identity->isAdmin();
                        }
                    ],
                ],
                'denyCallback' => function ($rule, $action) {
                    return $this->redirect(['/site/index']);
                }
            ],
        ];
    }

    /**
     * Главная страница личного кабинета
     */
    public function actionIndex()
    {
        $user = Yii::$app->user->identity;
        $profile = EmployeeProfile::findOne(['user_id' => $user->id]);

        return $this->render('index', [
            'user' => $user,
            'profile' => $profile,
        ]);
    }

    /**
     * Создание анкеты
     */
    public function actionCreate()
    {
        $user = Yii::$app->user->identity;
        
        // Проверяем, есть ли уже анкета
        $existingProfile = EmployeeProfile::findOne(['user_id' => $user->id]);
        if ($existingProfile) {
            Yii::$app->session->setFlash('info', 'Анкета уже заполнена. Вы можете ее редактировать.');
            return $this->redirect(['update']);
        }

        $model = new EmployeeProfile();
        $model->user_id = $user->id;

        if ($model->load(Yii::$app->request->post())) {
            $model->photoFile = UploadedFile::getInstance($model, 'photoFile');
            $model->passportScanFile = UploadedFile::getInstance($model, 'passportScanFile');
            $model->diplomaScanFile = UploadedFile::getInstance($model, 'diplomaScanFile');
            $model->snilsScanFile = UploadedFile::getInstance($model, 'snilsScanFile');

            if ($model->save()) {
                if ($model->upload()) {
                    Yii::$app->session->setFlash('success', 
                        '✅ Анкета успешно создана и отправлена на проверку!'
                    );
                } else {
                    Yii::$app->session->setFlash('warning', 
                        'Анкета сохранена, но возникли проблемы с загрузкой файлов.'
                    );
                }
                return $this->redirect(['index']);
            } else {
                Yii::$app->session->setFlash('error', 
                    'Ошибка при сохранении анкеты. Проверьте введенные данные.'
                );
            }
        }

        return $this->render('create', [
            'model' => $model,
            'user' => $user,
        ]);
    }

    /**
     * Редактирование анкеты
     */
    public function actionUpdate()
    {
        $user = Yii::$app->user->identity;
        $model = EmployeeProfile::findOne(['user_id' => $user->id]);

        if (!$model) {
            Yii::$app->session->setFlash('warning', 'Сначала создайте анкету.');
            return $this->redirect(['create']);
        }

        // Если анкета уже проверена администратором, запрещаем редактирование
        if ($model->status === EmployeeProfile::STATUS_ACCEPTED) {
            Yii::$app->session->setFlash('info', 
                'Анкета уже принята администратором. Для внесения изменений обратитесь к администратору.'
            );
            return $this->redirect(['index']);
        }

        if ($model->load(Yii::$app->request->post())) {
            $model->photoFile = UploadedFile::getInstance($model, 'photoFile');
            $model->passportScanFile = UploadedFile::getInstance($model, 'passportScanFile');
            $model->diplomaScanFile = UploadedFile::getInstance($model, 'diplomaScanFile');
            $model->snilsScanFile = UploadedFile::getInstance($model, 'snilsScanFile');

            // При редактировании сбрасываем статус на "новая"
            if ($model->status !== EmployeeProfile::STATUS_ACCEPTED) {
                $model->status = EmployeeProfile::STATUS_NEW;
                $model->reviewed_by = null;
                $model->reviewed_at = null;
                $model->review_comment = null;
            }

            if ($model->save()) {
                if ($model->upload()) {
                    Yii::$app->session->setFlash('success', 
                        '✅ Данные анкеты успешно обновлены и отправлены на повторную проверку!'
                    );
                } else {
                    Yii::$app->session->setFlash('warning', 
                        'Данные обновлены, но возникли проблемы с загрузкой файлов.'
                    );
                }
                return $this->redirect(['index']);
            } else {
                Yii::$app->session->setFlash('error', 
                    'Ошибка при обновлении анкеты. Проверьте введенные данные.'
                );
            }
        }

        return $this->render('update', [
            'model' => $model,
            'user' => $user,
        ]);
    }

    /**
     * Просмотр анкеты
     */
    public function actionView()
    {
        $user = Yii::$app->user->identity;
        $model = EmployeeProfile::findOne(['user_id' => $user->id]);

        if (!$model) {
            Yii::$app->session->setFlash('warning', 'Сначала создайте анкету.');
            return $this->redirect(['create']);
        }

        return $this->render('view', [
            'model' => $model,
            'user' => $user,
        ]);
    }

    /**
     * Находит модель анкеты по ID
     * @param integer $id
     * @return EmployeeProfile the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = EmployeeProfile::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException('Запрашиваемая страница не найдена.');
    }
}